import asyncio
import time
import statistics

from datetime import datetime

from pylog.pylogger import PyLogger

from py_pli.pylib import VUnits
from py_pli.pylib import GlobalVar

from urpc_enum.corexymoverparameter import CoreXYMoverParameter
from urpc_enum.fancontrolparameter import FanControlParameter
from urpc_enum.measurementparameter import MeasurementParameter
from urpc_enum.moverparameter import MoverParameter
from urpc_enum.nodeparameter import NodeParameter
from urpc_enum.serialparameter import SerialParameter
from urpc_enum.systemcontrolparameter import SystemControlParameter
from urpc_enum.temperaturecontrolparameter import TemperatureControlParameter

from fleming.common.firmware_util import *
from fleming.common.node_io import *

from virtualunits.vu_measurement_unit import VUMeasurementUnit
from virtualunits.meas_seq_generator import meas_seq_generator
from virtualunits.meas_seq_generator import TriggerSignal
from virtualunits.meas_seq_generator import OutputSignal
from virtualunits.meas_seq_generator import MeasurementChannel

from predefined_tasks.common.helper import send_to_gc


# From the terminal use the names defined in the endpoint dictionaries to access the endpoints directly.
def __getattr__(name):
    if name in node_endpoints:
        return get_node_endpoint(name)
    if name in mover_endpoints:
        return get_mover_endpoint(name)
    if name in stacker_lift_mover_endpoints:
        return get_stacker_lift_mover_endpoint(name)
    if name in corexy_mover_endpoints:
        return get_corexy_mover_endpoint(name)
    if name in measurement_endpoints:
        return get_measurement_endpoint(name)
    if name in serial_endpoints:
        return get_serial_endpoint(name)
    if name in system_control_endpoints:
        return get_system_control_endpoint(name)
    if name in fan_control_endpoints:
        return get_fan_control_endpoint(name)
    if name in temperature_control_endpoints:
        return get_temperature_control_endpoint(name)
    raise AttributeError(f"module '{__name__}' has no attribute '{name}'")


# Initialization #######################################################################################################

async def init(*endpoints):
    for endpoint in endpoints:
        if endpoint in node_endpoints:
            PyLogger.logger.info(f"Start '{endpoint}' firmware")
            await start_firmware(endpoint)
        if endpoint in system_control_endpoints:
            PyLogger.logger.info(f"Initialize '{endpoint}' System Control")
            sys = get_system_control_endpoint(endpoint)
            if button_event_callback not in sys._SystemControlFunctions__SendButtonEventEvents:
                sys.subscribeSendButtonEvent(button_event_callback)
            await sys.EnableSendButtonEvent(number=0, type=1, holdDelay=0)
            await sys.EnableSendButtonEvent(number=1, type=1, holdDelay=0)
            await sys.EnableSendButtonEvent(number=2, type=1, holdDelay=0)
            await sys.SetParameter(SystemControlParameter.ButtonLightBrightness,       100)
            await sys.SetParameter(SystemControlParameter.StatusLightGreenBrightness,  100)
            await sys.SetParameter(SystemControlParameter.StatusLightOrangeBrightness, 100)
            await sys.SetParameter(SystemControlParameter.ButtonLightBlinkPeriod,  500)
            await sys.SetParameter(SystemControlParameter.StatusLightBlinkPeriod, 1000)
        if endpoint in mover_endpoints:
            PyLogger.logger.info(f"Initialize '{endpoint}' Mover")
            mover = get_mover_endpoint(endpoint)
            if endpoint == 'as1' or endpoint == 'as2':
                await mover.SetProfile(
                    handle=0, speed=3000, accel=30000, decel=30000, uSteps=256, drivePower=32, holdPower=1, drivePowerHoldTime=1000, drivePowerFallTime=1000
                )
                await mover.UseProfile(0)
                await mover.SetParameter(MoverParameter.HomeSearchDirection,               0)
                await mover.SetParameter(MoverParameter.HomeMaxDistance,             1000000)
                await mover.SetParameter(MoverParameter.HomeMaxReverseDistance,      1000000)
                await mover.SetParameter(MoverParameter.HomeExtraReverseDistance,          0)
                await mover.SetParameter(MoverParameter.HomeCalibrationSpeed,          30000)
                await mover.SetParameter(MoverParameter.HomePosition,             0x7FFFFFFF)
                await mover.SetParameter(MoverParameter.HomeSensorEnable,               0x01)
                await mover.SetParameter(MoverParameter.MovementDirection,                 1)
                await mover.SetConfigurationStatus(1)
            elif endpoint == 'fm':
                await mover.SetProfile(
                    handle=0, speed=30000, accel=300000, decel=300000, uSteps=256, drivePower=10, holdPower=1, drivePowerHoldTime=1000, drivePowerFallTime=1000
                )
                await mover.UseProfile(0)
                await mover.SetParameter(MoverParameter.HomeSearchDirection,               0)
                await mover.SetParameter(MoverParameter.HomeMaxDistance,             1000000)
                await mover.SetParameter(MoverParameter.HomeMaxReverseDistance,      1000000)
                await mover.SetParameter(MoverParameter.HomeExtraReverseDistance,          0)
                await mover.SetParameter(MoverParameter.HomeCalibrationSpeed,          10000)
                await mover.SetParameter(MoverParameter.HomePosition,             0x7FFFFFFF)
                await mover.SetParameter(MoverParameter.HomeSensorEnable,               0x01)
                await mover.SetParameter(MoverParameter.MovementDirection,                 0)
                await mover.SetConfigurationStatus(1)
            elif endpoint == 'usfm':
                await mover.SetProfile(
                    handle=0, speed=30000, accel=300000, decel=300000, uSteps=256, drivePower=28, holdPower=14, drivePowerHoldTime=1000, drivePowerFallTime=1000
                )
                await mover.UseProfile(0)
                await mover.SetParameter(MoverParameter.HomeSearchDirection,               0)
                await mover.SetParameter(MoverParameter.HomeMaxDistance,             1000000)
                await mover.SetParameter(MoverParameter.HomeMaxReverseDistance,      1000000)
                await mover.SetParameter(MoverParameter.HomeExtraReverseDistance,          0)
                await mover.SetParameter(MoverParameter.HomeCalibrationSpeed,          10000)
                await mover.SetParameter(MoverParameter.HomePosition,             0x7FFFFFFF)
                await mover.SetParameter(MoverParameter.HomeSensorEnable,               0x01)
                await mover.SetParameter(MoverParameter.MovementDirection,                 0)
                await mover.SetConfigurationStatus(1)
            # elif endpoint == 'lrm' or endpoint == 'rrm':
            #     await mover.SetProfile(
            #         handle=0, speed=30000, accel=300000, decel=300000, uSteps=256, drivePower=12, holdPower=1, drivePowerHoldTime=1000, drivePowerFallTime=1000
            #     )
            #     await mover.UseProfile(0)
            #     await mover.SetParameter(MoverParameter.HomeSearchDirection,               0)
            #     await mover.SetParameter(MoverParameter.HomeMaxDistance,             1000000)
            #     await mover.SetParameter(MoverParameter.HomeMaxReverseDistance,      1000000)
            #     await mover.SetParameter(MoverParameter.HomeExtraReverseDistance,          0)
            #     await mover.SetParameter(MoverParameter.HomeCalibrationSpeed,          10000)
            #     await mover.SetParameter(MoverParameter.HomePosition,                 -35352)
            #     await mover.SetParameter(MoverParameter.HomeSensorEnable,               0x01)
            #     await mover.SetParameter(MoverParameter.MovementDirection,                 1)
            #     await mover.SetConfigurationStatus(1)
            elif endpoint == 'mc6m3':
                # Dispenser Hight Mover Test:
                await mover.SetProfile(
                    handle=0, speed=30000, accel=300000, decel=300000, uSteps=256, drivePower=42, holdPower=5, drivePowerHoldTime=1000, drivePowerFallTime=1000
                )
                await mover.UseProfile(0)
                await mover.SetParameter(MoverParameter.HomeSearchDirection,               0)
                await mover.SetParameter(MoverParameter.HomeMaxDistance,             1000000)
                await mover.SetParameter(MoverParameter.HomeMaxReverseDistance,      1000000)
                await mover.SetParameter(MoverParameter.HomeExtraReverseDistance,          0)
                await mover.SetParameter(MoverParameter.HomeCalibrationSpeed,          10000)
                await mover.SetParameter(MoverParameter.HomePosition,             0x7FFFFFFF)
                await mover.SetParameter(MoverParameter.HomeSensorEnable,               0x01)
                await mover.SetParameter(MoverParameter.MovementDirection,                 0)
                await mover.SetConfigurationStatus(1)
            else:
                await mover.SetProfile(
                    handle=0, speed=30000, accel=300000, decel=300000, uSteps=256, drivePower=100, holdPower=20, drivePowerHoldTime=1000, drivePowerFallTime=1000
                )
                await mover.UseProfile(0)
                await mover.SetParameter(MoverParameter.HomeSearchDirection,               0)
                await mover.SetParameter(MoverParameter.HomeMaxDistance,             1000000)
                await mover.SetParameter(MoverParameter.HomeMaxReverseDistance,      1000000)
                await mover.SetParameter(MoverParameter.HomeExtraReverseDistance,          0)
                await mover.SetParameter(MoverParameter.HomeCalibrationSpeed,          10000)
                await mover.SetParameter(MoverParameter.HomePosition,             0x7FFFFFFF)
                await mover.SetParameter(MoverParameter.HomeSensorEnable,               0x01)
                await mover.SetParameter(MoverParameter.MovementDirection,                 1)
                await mover.SetParameter(MoverParameter.EncoderMode,                       1)
                await mover.SetParameter(MoverParameter.EncoderCorrectionFactor,      25.600)
                await mover.SetParameter(MoverParameter.MaxEncoderDeviation,      0xFFFFFFFF)
                await mover.SetConfigurationStatus(1)
        if endpoint in stacker_lift_mover_endpoints:
            PyLogger.logger.info(f"Initialize '{endpoint}' StackerLiftMover")
            mover = get_stacker_lift_mover_endpoint(endpoint)
            await mover.SetProfile(
                handle=0, speed=30000, accel=300000, decel=300000, uSteps=256, drivePower=40, holdPower=20, drivePowerHoldTime=1000, drivePowerFallTime=1000
            )
            await mover.UseProfile(0)
            await mover.SetParameter(MoverParameter.HomeSearchDirection,               0)
            await mover.SetParameter(MoverParameter.HomeMaxDistance,             1000000)
            await mover.SetParameter(MoverParameter.HomeMaxReverseDistance,      1000000)
            await mover.SetParameter(MoverParameter.HomeExtraReverseDistance,          0)
            await mover.SetParameter(MoverParameter.HomeCalibrationSpeed,          10000)
            await mover.SetParameter(MoverParameter.HomePosition,                 -75598)
            await mover.SetParameter(MoverParameter.HomeSensorEnable,               0x01)
            await mover.SetParameter(MoverParameter.MovementDirection,                 0)
            await mover.SetConfigurationStatus(1)
        if endpoint in corexy_mover_endpoints:
            PyLogger.logger.info(f"Initialize '{endpoint}' CoreXY Mover")
            corexy = get_corexy_mover_endpoint(endpoint)
            await corexy.SetProfile(
                handle=0, speed=100000, accel=2000000, decel=2000000, uSteps=256, drivePower=40, holdPower=20, drivePowerHoldTime=1000, drivePowerFallTime=1000
            )
            await corexy.UseProfile(0)
            await corexy.SetParameter(CoreXYMoverParameter.RampMode,                           1)
            await corexy.SetParameter(CoreXYMoverParameter.HomeAxisOrder,                      1)
            await corexy.SetParameter(CoreXYMoverParameter.HomeSearchDirectionX,               0)
            await corexy.SetParameter(CoreXYMoverParameter.HomeSearchDirectionY,               0)
            await corexy.SetParameter(CoreXYMoverParameter.HomeMaxDistanceX,              550000)
            await corexy.SetParameter(CoreXYMoverParameter.HomeMaxDistanceY,              550000)
            await corexy.SetParameter(CoreXYMoverParameter.HomeMaxReverseDistanceX,        10000)
            await corexy.SetParameter(CoreXYMoverParameter.HomeMaxReverseDistanceY,        10000)
            await corexy.SetParameter(CoreXYMoverParameter.HomeExtraReverseDistanceX,          0)
            await corexy.SetParameter(CoreXYMoverParameter.HomeExtraReverseDistanceY,          0)
            await corexy.SetParameter(CoreXYMoverParameter.HomeCalibrationSpeed,           10000)
            await corexy.SetParameter(CoreXYMoverParameter.HomePositionX,             0x7FFFFFFF)
            await corexy.SetParameter(CoreXYMoverParameter.HomePositionY,             0x7FFFFFFF)
            await corexy.SetParameter(CoreXYMoverParameter.HomeSensorEnableX,               0x01)
            await corexy.SetParameter(CoreXYMoverParameter.HomeSensorEnableY,               0x01)
            await corexy.SetParameter(CoreXYMoverParameter.EncoderMode,                        1)
            await corexy.SetParameter(CoreXYMoverParameter.EncoderCorrectionFactorX,        12.8)
            await corexy.SetParameter(CoreXYMoverParameter.EncoderCorrectionFactorY,        12.8)
            await corexy.SetParameter(CoreXYMoverParameter.MaxEncoderDeviation,              512)
            await corexy.SetParameter(CoreXYMoverParameter.MovementDirectionX,                 0)
            await corexy.SetParameter(CoreXYMoverParameter.MovementDirectionY,                 0)
            await corexy.SetParameter(CoreXYMoverParameter.RotationalDirectionA,               0)
            await corexy.SetParameter(CoreXYMoverParameter.RotationalDirectionB,               0)
            await corexy.SetConfigurationStatus(1)
        if endpoint == 'alpha_tc':
            PyLogger.logger.info(f"Initialize Alpha Laser Temperature Control")
            tc = get_temperature_control_endpoint('eef_tc')
            channel = 0
            await tc.SetParameter(channel, TemperatureControlParameter.ReferenceValue, 0.33)
            await tc.SelectAnalogInput(channel, EEFAnalogInput.ALPHATEMPIN, scaling=1.0, offset=0.0)
            await tc.SelectAnalogOutput(channel, EEFAnalogOutput.ALPHATEC, scaling=1.0, offset=0.0)
            await tc.Configure(channel, dt=0.01, kp=-10.0, ti=float('inf'), td=0.0, min=0.0, max=+0.15)
            await tc.Enable(channel, enable=1)

    return f"init() done"


button_light_mode = [0, 0, 0]

async def button_event_callback(number, type):
    if number < len(button_light_mode):
        sys = get_system_control_endpoint()
        button_light_mode[number] = (button_light_mode[number] + 1) % 3
        await sys.SetButtonLight(number, button_light_mode[number])


# Mover Tests ##########################################################################################################

async def fragmented_move(mover_name, position, step_size=1, timeout=1):
    mover = get_mover_endpoint(mover_name)
    current_position = (await mover.GetPosition())[0]
    PyLogger.logger.info(f"current_position = {current_position}")
    if (position >= current_position):
        dir = +1
    else:
        dir = -1
    for pos in range((current_position + (step_size * dir)), (position + dir), (step_size * dir)):
        PyLogger.logger.info(f"Move({pos})")
        await mover.Move(pos, timeout)
    
    return f"fragmented_move() done"


async def as_homing_test(as_number=1, iterations=1):
    eef_unit = VUnits.instance.hal.nodes['EEFNode']
    if as_number == 1:
        as_unit = VUnits.instance.hal.detectorApertureSlider1
    if as_number == 2:
        as_unit = VUnits.instance.hal.detectorApertureSlider2

    for i in range(iterations):
        await eef_unit.node.Reset(timeout=1)
        await asyncio.sleep(0.1)
        await eef_unit.StartFirmware()
        await as_unit.InitializeDevice()
        step_error = await as_unit.Home()
        PyLogger.logger.info(f"{i+1}: step_error={step_error}")

    return f"as_homing_test() done"


async def fpga_init_test(iterations=1):
    eef = get_node_endpoint('eef')

    for i in range(iterations):
        PyLogger.logger.info(f"iteration: {i+1}")
        await eef.Reset(timeout=1)
        await asyncio.sleep(0.1)
        await start_firmware('eef')
        await VUnits.instance.hal.detectorApertureSlider1.GetPosition()
        await VUnits.instance.hal.detectorApertureSlider2.GetPosition()

    return f"fpga_init_test() done"


async def fpga_write_seq_test(iterations=1):
    meas = get_measurement_endpoint()

    start = 0
    for i in range(iterations):
        buffer = range(start, (start + 28))
        start = start + len(buffer)
        PyLogger.logger.info(f"WriteSequence: {list(buffer)}")
        await meas.WriteSequence(0, len(buffer), buffer, timeout=5)
        
    return f"fpga_write_test() done"


async def fpga_write_pos_test(iterations=1):
    as1 = get_mover_endpoint('as1')
    as2 = get_mover_endpoint('as2')
    for i in range(iterations):
        await as1.SetPosition(i)
        await as2.SetPosition(i)
        PyLogger.logger.info(f"Pos: {(await as1.GetPosition(timeout=1))[0]}, {(await as2.GetPosition(timeout=1))[0]}")
        
    return f"fpga_write_test() done"


async def fpga_mem_test(start_address=0x2000, stop_address=0x5000, iterations=1):
    start_address = int(start_address, 0)
    stop_address = int(stop_address, 0)
    meas = get_measurement_endpoint()
    success = True
    for i in range(iterations):
        PyLogger.logger.info(f"fpga_mem_test {i} - started")
        size = 56
        for address in range(start_address, stop_address, size):
            if (address + size) > stop_address:
                size = stop_address - address
            await meas.Write(address, size, [0xFFFF] * size)
        size = 56
        for address in range(start_address, stop_address, size):
            if (address + size) > stop_address:
                size = stop_address - address
            data_block = await meas.Read(address, size)
            for offset, data in enumerate(data_block):
                if data != 0xFFFF:
                    success = False
                    PyLogger.logger.info(f"fpga_mem_test {i} - address 0x{address + offset:04X} failed write")
        size = 56
        for address in range(start_address, stop_address, size):
            if (address + size) > stop_address:
                size = stop_address - address
            await meas.Write(address, size, [0] * size)
        size = 56
        for address in range(start_address, stop_address, size):
            if (address + size) > stop_address:
                size = stop_address - address
            data_block = await meas.Read(address, size)
            for offset, data in enumerate(data_block):
                if data != 0:
                    success = False
                    PyLogger.logger.info(f"fpga_mem_test {i} - address 0x{address + offset:04X} failed clear")

        PyLogger.logger.info(f"fpga_mem_test {i} - completed")
    
    return f"fpga_mem_test {'passed' if success else 'failed'}"


async def fpga_seq_mem_test(iterations=1):
    meas_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit

    for i in range(iterations):
        sequence = [0xFFFFFFFF] * 0x0401
        meas_unit.ClearOperations()
        await meas_unit.LoadTriggerSequence('seq_mem_set', sequence)
        sequence = [0] * 0x0400
        meas_unit.ClearOperations()
        await meas_unit.LoadTriggerSequence('seq_mem_clear', sequence)
        
    return f"fpga_seq_mem_test done"


# Sequencer Tests ######################################################################################################

signals = {
    'flash'     : (1 << 23),
    'alpha'     : (1 << 22),
    'htsalpha'  : (1 << 21),
    'ingate2'   : (1 << 17),
    'ingate1'   : (1 << 16),
    'hvgate2'   : (1 << 13),
    'hvgate1'   : (1 << 12),
    'hvon3'     : (1 << 10),
    'hvon2'     : (1 <<  9),
    'hvon1'     : (1 <<  8),
    'rstaux'    : (1 <<  6),
    'rstabs'    : (1 <<  5),
    'rstref'    : (1 <<  4),
    'rstpmt2'   : (1 <<  1),
    'rstpmt1'   : (1 <<  0),
}

triggers = {
    'trf'   : (1 << 8),
    'aux'   : (1 << 6),
    'abs'   : (1 << 5),
    'ref'   : (1 << 4),
    'pmt3'  : (1 << 2),
    'pmt2'  : (1 << 1),
    'pmt1'  : (1 << 0),
}

channels = {
    'pmt1'  : (0 << 24),
    'pmt2'  : (1 << 24),
    'pmt3'  : (2 << 24),
    'pmt4'  : (3 << 24),
    'ref'   : (4 << 24),
    'abs'   : (5 << 24),
    'aux'   : (6 << 24),
}


async def seq_pulse_signal(name, duration_us=1):
    meas = get_measurement_endpoint()
    sequence = [
        0x7C000000 | (duration_us * 100 - 1),   # Timer Wait And Restart
        0x02000000 | signals[name],             # Set Signal
        0x7C000000,                             # Timer Wait
        0x03000000 | signals[name],             # Reset Signal
        0x00000000,
    ]
    await meas.WriteSequence(0, len(sequence), sequence, timeout=5)
    await meas.StartSequence(0)
    
    return f"seq_pulse_signal() done"


async def seq_set_signal(name):
    meas = get_measurement_endpoint()
    sequence = [
        0x02000000 | signals[name],     # Set Signal
        0x00000000,
    ]
    await meas.WriteSequence(0, len(sequence), sequence, timeout=5)
    await meas.StartSequence(0)
    
    return f"seq_set_signal() done"


async def seq_clear_signal(name):
    meas = get_measurement_endpoint()
    sequence = [
        0x03000000 | signals[name],     # Reset Signal
        0x00000000,
    ]
    await meas.WriteSequence(0, len(sequence), sequence, timeout=5)
    await meas.StartSequence(0)
    
    return f"seq_clear_signal() done"


async def seq_trigger(name):
    meas = get_measurement_endpoint()
    sequence = [
        0x01000000 | triggers[name],    # Set Trigger Output
        0x00000000,
    ]
    await meas.WriteSequence(0, len(sequence), sequence, timeout=5)
    await meas.StartSequence(0)
    
    return f"seq_trigger() done"


async def seq_wait_for_trigger(timeout=5):
    meas = get_measurement_endpoint()
    sequence = [
        0x74000000 | 1,                 # Wait For Trigger Input
        0x00000000,
    ]
    await meas.WriteSequence(0, len(sequence), sequence, timeout=5)
    await meas.StartSequence(0)
    PyLogger.logger.info(f"waiting for trigger...")
    start = time.perf_counter()
    while not (await meas.GetStatus())[0] & 0x01:
        if (time.perf_counter() - start) > timeout:
            raise Exception(f"seq_wait_for_trigger() did timeout")

    return f"waiting for trigger done"


async def seq_trf_loopback():
    meas = get_measurement_endpoint()
    sequence = [
        0x01000000 | triggers['trf'],   # Set Trigger Output
        0x74000000 | 1,                 # Wait For Trigger Input
        0x00000000,
    ]
    await meas.WriteSequence(0, len(sequence), sequence, timeout=5)
    await meas.StartSequence(0)
    await asyncio.sleep(0.1)
    if (await meas.GetStatus())[0] & 0x01:
        return f"seq_trf_loopback() passed"
    else:
        return f"seq_trf_loopback() failed"
            

async def seq_get_analog(detector='pmt1', window_us=1000):
    meas = get_measurement_endpoint()
    window_corse, window_fine = divmod(window_us, 65536)
    reset_delay = 1000
    sequence = [
        0x7C000000 | (reset_delay - 1),         # Timer Start Reset Delay
        0x02000000 | signals['rst' + detector], # Set Reset Signal
        0x8C040000,                             # Clear Result Buffer
        0x7C000000 | (100 - 1),                 # Timer Wait And Restart 1 µs
        0x03000000 | signals['rst' + detector], # Clear Reset Signal
    ]
    if window_corse > 0:
        sequence.extend([
            0x07000000 | (window_corse - 1),    # Loop for window_corse * 65536 µs
            0x07000000 | (65536 - 1),           # Loop for 65536 * 1 µs
            0x7C000000 | (100 - 1),             # Timer Wait And Restart 1 µs
            0x05000000,                         # Loop End
            0x05000000,                         # Loop End
        ])
    if window_fine > 0:
        sequence.extend([
            0x07000000 | (window_fine - 1),     # Loop for window_fine * 1 µs
            0x7C000000 | (100 - 1),             # Timer Wait And Restart 1 µs
            0x05000000,                         # Loop End
        ])
    sequence.extend([
        0x01000000 | triggers[detector],        # Trigger Analog Measurement
        0x80000000 | channels[detector],        # Get Analog High Result
        0x80100001 | channels[detector],        # Get Analog Low Result
        0x00000000,
    ])
    await meas.WriteSequence(0, len(sequence), sequence, timeout=5)
    await meas.StartSequence(0)
    results = await meas.ReadResults(0, 4, timeout=5)
    
    PyLogger.logger.info(f"{detector} ; analog_low = {5 / 65536 * results[0]} V ; analog_high = {5 / 65536 * results[1]} V")

    return results


async def seq_get_counter(detector='pmt1', window_us=100):
    meas = get_measurement_endpoint()
    window_corse, window_fine = divmod(window_us, 65536)
    sequence = [
        0x7C000000 | (100 - 1),                 # Timer Wait And Restart 1 µs Precounter Window
        0x88B80000 | channels[detector],        # Reset Precounter and Counter
    ]
    if window_corse > 0:
        sequence.extend([
            0x07000000 | (window_corse - 1),    # Loop for window_corse * 65536 µs
            0x07000000 | (65536 - 1),           # Loop for 65536 * 1 µs
            0x7C000000 | (100 - 1),             # Timer Wait And Restart 1 µs Precounter Window
            0x88D00000 | channels[detector],    # Add Precounter Value to the Counter Value
            0x05000000,                         # Loop End
            0x05000000,                         # Loop End
        ])
    if window_fine > 0:
        sequence.extend([
            0x07000000 | (window_fine - 1),     # Loop for window_fine * 1 µs
            0x7C000000 | (100 - 1),             # Timer Wait And Restart 1 µs Precounter Window
            0x88D00000 | channels[detector],    # Add Precounter Value to the Counter Value
            0x05000000,                         # Loop End
        ])
    sequence.extend([
        0x80900000 | channels[detector],        # Get Pulse Counter Result
        0x00000000,
    ])
    await meas.WriteSequence(0, len(sequence), sequence, timeout=5)
    await meas.StartSequence(0)
    results = await meas.ReadResults(0, 1, timeout=5)
    
    PyLogger.logger.info(f"Count: {detector} = {results[0]:,}")

    return results


async def seq_measure(window_us=1000, iterations=1):
    meas = get_measurement_endpoint()
    window_corse, window_fine = divmod(window_us, 65536)
    gate_delay = 1000000  # 10 ms
    sequence = [
        0x7C000000 | (gate_delay - 1),                          # Timer Start Gate Delay
        0x02000000 | signals['hvgate1'] | signals['hvgate2']    # Set Signals 
                   | signals['ingate1'] | signals['ingate2'] 
                   | signals['rstpmt1'] | signals['rstpmt2'],
        0x8C000000,                                             # Clear Result Buffer 0
        0x8C000001,                                             # Clear Result Buffer 1
        0x8C000002,                                             # Clear Result Buffer 2
        0x8C000003,                                             # Clear Result Buffer 3
        0x8C000004,                                             # Clear Result Buffer 4
        0x8C000005,                                             # Clear Result Buffer 5
        0x7C000000 | (100 - 1),                                 # Timer Wait And Restart 1 µs Precounter Window
        0x03000000 | signals['rstpmt1'] | signals['rstpmt2'],   # Reset Signals
        0x88B80000 | channels['pmt1'],                          # Reset Precounter and Counter
        0x88B80000 | channels['pmt2'],                          # Reset Precounter and Counter
    ]
    if window_corse > 0:
        sequence.extend([
            0x07000000 | (window_corse - 1),                    # Loop for window_corse * 65536 µs
            0x07000000 | (65536 - 1),                           # Loop for 65536 * 1 µs
            0x7C000000 | (100 - 1),                             # Timer Wait And Restart 1 µs Precounter Window
            0x88D00000 | channels['pmt1'],                      # Add Precounter Value to the Counter Value
            0x88D00000 | channels['pmt2'],                      # Add Precounter Value to the Counter Value
            0x05000000,                                         # Loop End
            0x05000000,                                         # Loop End
        ])
    if window_fine > 0:
        sequence.extend([
            0x07000000 | (window_fine - 1),                     # Loop for window_fine * 1 µs
            0x7C000000 | (100 - 1),                             # Timer Wait And Restart 1 µs Precounter Window
            0x88D00000 | channels['pmt1'],                      # Add Precounter Value to the Counter Value
            0x88D00000 | channels['pmt2'],                      # Add Precounter Value to the Counter Value
            0x05000000,                                         # Loop End
        ])
    sequence.extend([
        0x01000000 | triggers['pmt1'] | triggers['pmt2'],       # Trigger Analog Measurement
        0x80900000 | channels['pmt1'],                          # Get Pulse Counter Result
        0x80000001 | channels['pmt1'],                          # Get Analog High Result
        0x80100002 | channels['pmt1'],                          # Get Analog Low Result
        0x80900003 | channels['pmt2'],                          # Get Pulse Counter Result
        0x80000004 | channels['pmt2'],                          # Get Analog High Result
        0x80100005 | channels['pmt2'],                          # Get Analog Low Result
        0x03000000 | signals['hvgate1'] | signals['hvgate2']    # Reset Signals
                   | signals['ingate1'] | signals['ingate2'],
        0x00000000,
    ])
    for i in range(0, len(sequence), 28):
        subsequence = sequence[i:(i + 28)]
        await meas.WriteSequence(i, len(subsequence), subsequence, timeout=5)

    results = [[], [], [], [], [], []]
    for i in range(iterations):
        PyLogger.logger.info(f"seq_measure(): iteration = {i}")
        await meas.StartSequence(0)
        result = await meas.ReadResults(0, 6, timeout=(2 + window_us / 1000000))
        results[0].append(result[0])
        results[1].append(5 / 65536 * result[1])
        results[2].append(5 / 65536 * result[2])
        results[3].append(result[3])
        results[4].append(5 / 65536 * result[4])
        results[5].append(5 / 65536 * result[5])
    
        PyLogger.logger.info(f"pmt1 ; count = {results[0][i]:,} ; analog_low = {results[1][i]} V ; analog_high = {results[2][i]} V")
        PyLogger.logger.info(f"pmt2 ; count = {results[3][i]:,} ; analog_low = {results[4][i]} V ; analog_high = {results[5][i]} V")

    return results


async def seq_darkcount(iterations, delay=100000):
    meas = get_measurement_endpoint()
    sequence = [
        0x7C000000 | (delay - 1),           # TimerWaitAndRestart(delay - 1)
        0x88B80000,                         # PulseCounterControl(ch=0, add=0, RstCnt=1, RstPreCnt=1, corr=1)
        0x07000000 | (iterations - 1),      # Loop(iterations - 1)
        0x7C000000 | (delay - 1),           #     TimerWaitAndRestart(delay - 1)
        0x88D00000,                         #     PulseCounterControl(ch=0, add=1, RstCnt=0, RstPreCnt=1, corr=0)
        0x05000000,                         # LoopEnd()
        0x80900000,                         # GetPulseCounterResult(ch=0, rel=0, RstCnt=1, add=0, dword=0, addrReg=0, addr=0)
        0x00000000,                         # Stop(0)
    ]
    await meas.WriteSequence(0, len(sequence), sequence, timeout=5)
    await meas.StartSequence(0)
    results = await meas.ReadResults(0, 1, timeout=5)
    
    PyLogger.logger.info(f"Dark Count = {results[0]:,}")


# Example for the On-The-Fly synchonisation of the measurement sequence and the scan table.
# The sequence generates 50% duty cycle pulses for every full step of the scan table.
# The number of microsteps per full step can be freely chosen.
async def seq_on_the_fly(full_steps, micro_steps_per_full_step=256):
    meas = get_measurement_endpoint()
    corexy = get_corexy_mover_endpoint()
    full_step_cnt = micro_steps_per_full_step * 4   # The stepcounter counts in quarter microsteps!
    sequence = [
        0x07000000 | (full_steps - 1),          # Loop Full Steps
        0x6C000000 | int(full_step_cnt / 4),    # Stepcounter Wait and Restart Quarter Step
        0x03000000 | signals['flash'],          # Reset Signal
        0x6C000000 | int(full_step_cnt / 2),    # Stepcounter Wait and Restart Half Step
        0x02000000 | signals['flash'],          # Set Signal
        0x6C000000 | int(full_step_cnt / 4),    # Stepcounter Wait and Restart Quarter Step
        0x03000000 | signals['flash'],          # Reset Signal
        0x05000000,                             # Loop End
        0x00000000,
    ]
    await meas.WriteSequence(0, len(sequence), sequence, timeout=5)

    # Home and move to start position
    await corexy.Home()
    await corexy.Move(0, 0)
    
    micro_steps = full_steps * micro_steps_per_full_step

    # Move in positive x direction
    await meas.SetParameter(MeasurementParameter.StepCounterSelect, 0x01)
    await meas.StartSequence(0)
    await corexy.Move(micro_steps, 0)

    # Move in positive y direction
    await meas.SetParameter(MeasurementParameter.StepCounterSelect, 0x00)
    await meas.StartSequence(0)
    await corexy.Move(micro_steps, micro_steps)

    # Move in negative x direction
    await meas.SetParameter(MeasurementParameter.StepCounterSelect, 0x81)
    await meas.StartSequence(0)
    await corexy.Move(0, micro_steps)

    # Move in negative y direction
    await meas.SetParameter(MeasurementParameter.StepCounterSelect, 0x80)
    await meas.StartSequence(0)
    await corexy.Move(0, 0)
    
    return f"seq_on_the_fly() done"


async def seq_retrigger_test(detector='ref', delay_us=9, duration_ms=0.15):
    trigger = {'pmt1':TriggerSignal.SamplePMT1, 'pmt2':TriggerSignal.SamplePMT2, 'ref':TriggerSignal.SampleRef, 'abs':TriggerSignal.SampleRef, 'aux':TriggerSignal.SampleAux}
    meas_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit

    delay = round(delay_us * 100)
    iterations = round(duration_ms * 1e5 / delay)

    op_id = 'seq_convst_retrigger_test'
    seq_gen = meas_seq_generator()
    seq_gen.Loop(iterations)
    seq_gen.TimerWaitAndRestart(delay)
    seq_gen.SetTriggerOutput(trigger[detector])
    seq_gen.LoopEnd()
    seq_gen.Stop(0)

    meas_unit.ClearOperations()
    await meas_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
    await meas_unit.ExecuteMeasurement(op_id)

    return f"seq_convst_retrigger_test() done"


# PD Tests #############################################################################################################

async def pd_ref_test_v1(start_us=20, stop_us=100, step_us=1, fixed_range_us=5):
    meas_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    full_reset_delay = 40000    # 400 us
    low_reset_delay  =  1000    #  10 us

    if (start_us <= 0) or (stop_us <= 0) or (step_us <= 0) or (start_us > stop_us) or (fixed_range_us < 0) or (fixed_range_us > start_us):
        raise ValueError(f"Illegal Argument")

    timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
    with open(f"pd_ref_test_v1_{timestamp}.txt", 'w') as file:
        file.write(f"pd_ref_test_v1(start_us={start_us}, stop_us={stop_us}, step_us={step_us}, fixed_range_us={fixed_range_us})\n")
        file.write(f"time [ms]   ; analog_low  ; analog_high\n")
        for window in range(round(start_us * 100), round(stop_us * 100 + 1), round(step_us * 100)):
            fixed_range = round(fixed_range_us * 100)
            op_id = 'pd_ref_test_v1'
            seq_gen = meas_seq_generator()
            seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=0)
            seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=1)
            seq_gen.TimerWaitAndRestart(full_reset_delay)
            seq_gen.currSequence.append(0x0F000000 | (2 << 12))     # SetIntegratorMode(IntModeRef=full_reset)
            seq_gen.TimerWaitAndRestart(low_reset_delay)
            seq_gen.currSequence.append(0x0F000000 | (3 << 12))     # SetIntegratorMode(IntModeRef=low_reset)
            seq_gen.TimerWaitAndRestart(window - fixed_range)
            seq_gen.currSequence.append(0x0F000000 | (4 << 12))     # SetIntegratorMode(IntModeRef=auto_range)
            if fixed_range > 0:
                seq_gen.TimerWaitAndRestart(fixed_range)
                seq_gen.currSequence.append(0x0F000000 | (5 << 12))     # SetIntegratorMode(IntModeRef=fixed_range)
            seq_gen.TimerWait()
            seq_gen.SetTriggerOutput(TriggerSignal.SampleRef)
            seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=False, dword=False, addrPos=0, resultPos=0)
            seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=False, dword=False, addrPos=0, resultPos=1)
            seq_gen.currSequence.append(0x0F000000 | (2 << 12))     # SetIntegratorMode(IntModeRef=full_reset)
            seq_gen.Stop(0)
            meas_unit.ClearOperations()
            meas_unit.resultAddresses[op_id] = range(0, 2)
            await meas_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
            await meas_unit.ExecuteMeasurement(op_id)
            results = await meas_unit.ReadMeasurementValues(op_id)

            file.write(f"{(window / 100000):11.4f} ; {results[0]:11d} ; {results[1]:11d}\n")

    return f"pd_ref_test_v1() done"
    

async def pd_ref_test_v2(window_us=20, iterations=100, delay=0):
    meas_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    full_reset_delay = 40000    # 400 us
    low_reset_delay  =  1000    #  10 us

    if (window_us <= 0) or (iterations <= 0):
        raise ValueError(f"Illegal Argument")

    timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
    with open(f"pd_ref_test_v2_{timestamp}.txt", 'w') as file:
        file.write(f"pd_ref_test_v2(window_us={window_us}, iterations={iterations})\n")
        file.write(f"time [ms]   ; analog_low  ; analog_high\n")
        for i in range(iterations):
            window = round(window_us * 100)
            op_id = 'pd_ref_test_v2'
            seq_gen = meas_seq_generator()
            seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=0)
            seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=1)
            seq_gen.TimerWaitAndRestart(full_reset_delay)
            seq_gen.currSequence.append(0x0F000000 | (2 << 12))     # SetIntegratorMode(IntModeRef=full_reset)
            seq_gen.TimerWaitAndRestart(low_reset_delay)
            seq_gen.currSequence.append(0x0F000000 | (3 << 12))     # SetIntegratorMode(IntModeRef=low_reset)
            seq_gen.TimerWaitAndRestart(window)
            seq_gen.currSequence.append(0x0F000000 | (6 << 12))     # SetIntegratorMode(IntModeRef=low_range)
            seq_gen.TimerWait()
            seq_gen.SetTriggerOutput(TriggerSignal.SampleRef)
            seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=False, dword=False, addrPos=0, resultPos=0)
            seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=False, dword=False, addrPos=0, resultPos=1)
            seq_gen.currSequence.append(0x0F000000 | (2 << 12))     # SetIntegratorMode(IntModeRef=full_reset)
            seq_gen.Stop(0)
            meas_unit.ClearOperations()
            meas_unit.resultAddresses[op_id] = range(0, 2)
            await meas_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
            await meas_unit.ExecuteMeasurement(op_id)
            results = await meas_unit.ReadMeasurementValues(op_id)

            file.write(f"{i:6d} ; {results[0]:11d} ; {results[1]:11d}\n")
            if delay > 0:
                await asyncio.sleep(delay)

    return f"pd_ref_test_v2() done"


async def pd_int_test(pd_name, sample_rate=4000, sample_count=100):
    pd_ctrl = {
        'ref': {'rst': OutputSignal.IntRstRef, 'trg': TriggerSignal.SampleRef, 'chn': 4}, 
        'abs': {'rst': OutputSignal.IntRstAbs, 'trg': TriggerSignal.SampleAbs, 'chn': 5}, 
        'aux': {'rst': OutputSignal.IntRstAux, 'trg': TriggerSignal.SampleAux, 'chn': 6},
    }
    if pd_name not in pd_ctrl:
        raise ValueError(f"pd_name must be in {pd_ctrl.keys()}")
    if (sample_rate < 10) or (sample_rate > 4000):
        raise ValueError(f"sample_rate must be in the range [10, 4000] Hz")
    if (sample_count < 1) or (sample_count > 500):
        raise ValueError(f"sample_count must be in the range [1, 500]")
    
    meas_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit

    reset_delay = 100000                            # 1 ms
    sample_delay = round(100000000 / sample_rate)   # 10 ns resolution
    result_count = sample_count * 2

    seq_gen = meas_seq_generator()
    # Clear the result buffer
    seq_gen.SetAddrReg(relative=False, dataNotAddrSrc=False, sign=False, stackNotRegSrc=False, srcReg=0, dstReg=0, addr=0)
    seq_gen.Loop(result_count)
    seq_gen.ClearResultBuffer(relative=True, dword=False, addrReg=0, addr=0)
    seq_gen.SetAddrReg(relative=True, dataNotAddrSrc=False, sign=False, stackNotRegSrc=False, srcReg=0, dstReg=0, addr=1)
    seq_gen.LoopEnd()
    # Set the integrator reset
    seq_gen.TimerWaitAndRestart(reset_delay)
    seq_gen.SetSignals(pd_ctrl[pd_name]['rst'])
    # Clear the integrator reset
    seq_gen.TimerWaitAndRestart(sample_delay)
    seq_gen.ResetSignals(pd_ctrl[pd_name]['rst'])
    # Sample the integrator curve
    seq_gen.SetAddrReg(relative=False, dataNotAddrSrc=False, sign=False, stackNotRegSrc=False, srcReg=0, dstReg=0, addr=0)
    seq_gen.Loop(sample_count)
    seq_gen.SetTriggerOutput(pd_ctrl[pd_name]['trg'])
    seq_gen.GetAnalogResult(channel=pd_ctrl[pd_name]['chn'], isRelativeAddr=True, ignoreRange=False, isHiRange=False, addResult=False, dword=False, addrPos=0, resultPos=0)
    seq_gen.GetAnalogResult(channel=pd_ctrl[pd_name]['chn'], isRelativeAddr=True, ignoreRange=False, isHiRange=True,  addResult=False, dword=False, addrPos=0, resultPos=1)
    seq_gen.TimerWaitAndRestart(sample_delay)
    seq_gen.SetAddrReg(relative=True, dataNotAddrSrc=False, sign=False, stackNotRegSrc=False, srcReg=0, dstReg=0, addr=2)
    seq_gen.LoopEnd()
    # Set the integrator reset and stop
    seq_gen.SetSignals(pd_ctrl[pd_name]['rst'])
    seq_gen.Stop(0)

    op_id = 'pd_int_test'

    meas_unit.ClearOperations()
    meas_unit.resultAddresses[op_id] = range(0, result_count)
    await meas_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
    await meas_unit.ExecuteMeasurement(op_id)

    results = await meas_unit.ReadMeasurementValues(op_id)

    timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
    with open(f"pd_int_test_{timestamp}.txt", 'w') as file:
        file.write(f"pd_int_test(pd_name={pd_name}, sample_rate={sample_rate}, sample_count={sample_count})\n")
        file.write(f"time [ms]   ; analog_low  ; analog_high\n")
        time_ms = 0.0
        for i in range(sample_count):
            file.write(f"{time_ms:11.3f} ; {results[i * 2 + 0]:11d} ; {results[i * 2 + 1]:11d}\n")
            time_ms = time_ms + sample_delay / 100000.0

    return f"pd_int_test() done"


# PMT Tests ############################################################################################################

async def set_pmt_hv(pmt, value):
    meas = get_measurement_endpoint()
    if pmt == 'pmt1' or pmt == 1:
        await meas.SetParameter(MeasurementParameter.PMT1HighVoltageSetting, value)
    if pmt == 'pmt2' or pmt == 2:
        await meas.SetParameter(MeasurementParameter.PMT2HighVoltageSetting, value)
    if pmt == 'pmtus' or pmt == 'pmt3' or pmt == 3:
        await meas.SetParameter(MeasurementParameter.PMTUSLUMHighVoltageSetting, value)
    
    return f"set_pmt_hv() done"


async def set_pmt_dl(pmt, value):
    meas = get_measurement_endpoint()
    if pmt == 'pmt1' or pmt == 1:
        await meas.SetParameter(MeasurementParameter.PMT1DiscriminatorLevel, value)
    if pmt == 'pmt2' or pmt == 2:
        await meas.SetParameter(MeasurementParameter.PMT2DiscriminatorLevel, value)
    if pmt == 'pmtus' or pmt == 'pmt3' or pmt == 3:
        await meas.SetParameter(MeasurementParameter.PMTUSLUMDiscriminatorLevel, value)
    
    return f"set_pmt_dl() done"


async def pmt_dl_sweep(pmt_name, window_us=1000000, iterations=10, dl_from=0.0, dl_to=1.0, dl_step=0.1, log_raw=True):
    precision = 1000.0
    dl_range = [dl / precision for dl in range(round(dl_from * precision), round(dl_to * precision + 1), round(dl_step * precision))]
    duration = (window_us / 1000000.0 + 0.03) * iterations * len(dl_range)
    print(f"duration = ~{duration:.0f} s")
    comment = input(f"Enter a comment ('X' to cancel): ")
    if comment == 'X' or comment == 'x':
        return f"pmt_dl_sweep() canceled"

    if pmt_name == 'pmt1':
        dl_parameter = MeasurementParameter.PMT1DiscriminatorLevel
        results_offset = 0
    if pmt_name == 'pmt2':
        dl_parameter = MeasurementParameter.PMT2DiscriminatorLevel
        results_offset = 3
    
    meas = get_measurement_endpoint()
    start = time.perf_counter()
    timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
    with open(f"pmt_dl_sweep_{timestamp}.txt", 'w') as file:
        file.write(f"pmt_dl_sweep(pmt_name={pmt_name}, window_us={window_us}, iterations={iterations}, dl_from={dl_from}, dl_to={dl_to}, dl_step={dl_step}, log_raw={log_raw})\n")
        file.write(f"Comment: {comment}\n")
        file.write(f"discriminator_level ; ")
        file.write(f"count_mean          ; count_min           ; count_max           ; count_stdev         ; ")
        file.write(f"analog_low_mean     ; analog_low_min      ; analog_low_max      ; analog_low_stdev    ; ")
        file.write(f"analog_high_mean    ; analog_high_min     ; analog_high_max     ; analog_high_stdev   ; ")
        file.write(f"\n")
        for dl in dl_range:
            PyLogger.logger.info(f"pmt_dl_sweep(): discriminator_level = {dl:.4f}")
            await meas.SetParameter(dl_parameter, dl)
            results = (await seq_measure(window_us, iterations))[results_offset:(results_offset + 3)]
            file.write(f"{dl:19.4f} ; ")
            file.write(f"{statistics.mean(results[0]):19.1f} ; {min(results[0]):19.1f} ; {max(results[0]):19.1f} ; {statistics.stdev(results[0]):19.1f} ; ")
            file.write(f"{statistics.mean(results[1]):19.6f} ; {min(results[1]):19.6f} ; {max(results[1]):19.6f} ; {statistics.stdev(results[1]):19.6f} ; ")
            file.write(f"{statistics.mean(results[2]):19.6f} ; {min(results[2]):19.6f} ; {max(results[2]):19.6f} ; {statistics.stdev(results[2]):19.6f} ; ")
            file.write(f"\n")
            if log_raw:
                with open(f"pmt_dl_sweep_raw_{timestamp}.txt", 'a') as raw_file:
                    raw_file.write(f"{dl}, {results}\n")

    return f"pmt_dl_sweep() done in {time.perf_counter() - start:.0f} s"


# Serial Tests #########################################################################################################

async def serial_loopback(serial_name, *txdata) -> bytes:
    serial = get_serial_endpoint(serial_name)
    await serial.SetParameter(SerialParameter.ReadTimeout, 1900)
    await serial.ClearReadBuffer(timeout=1)
    await serial.Write(len(txdata), bytes(txdata), timeout=2)
    rxdata = None
    try:
        response = await serial.Read(len(txdata), timeout=2)
        length = response[0]
        rxdata = response[1:(1 + length)]
    except BaseException as ex:
        PyLogger.logger.error(ex)

    PyLogger.logger.info(f"rxdata: {rxdata}")
    return rxdata


# BCR Tests ############################################################################################################

async def bcr_write(bcr_name, *data):
    bcr = get_serial_endpoint(bcr_name)
    node = get_node_endpoint(serial_endpoints[bcr_name]['node'])
    if 'mux' in serial_endpoints[bcr_name]:
        mux = serial_endpoints[bcr_name]['mux']
        await node.SetDigitalOutput(EEFDigitalOutput.BCREXTMUX0, bool(mux & 1))
        await node.SetDigitalOutput(EEFDigitalOutput.BCREXTMUX1, bool(mux & 2))
        await node.SetDigitalOutput(EEFDigitalOutput.BCREXTMUXEN, 0)

    await bcr.Write(len(data), bytes(data))

    return f"bcr_write() done"


async def bcr_read(bcr_name, length, timeout=5, sw_trigger=False) -> bytes:
    bcr = get_serial_endpoint(bcr_name)
    node = get_node_endpoint(serial_endpoints[bcr_name]['node'])
    ntrig = serial_endpoints[bcr_name]['ntrig']
    barcode = None
    if 'mux' in serial_endpoints[bcr_name]:
        mux = serial_endpoints[bcr_name]['mux']
        await node.SetDigitalOutput(EEFDigitalOutput.BCREXTMUX0, bool(mux & 1))
        await node.SetDigitalOutput(EEFDigitalOutput.BCREXTMUX1, bool(mux & 2))
        await node.SetDigitalOutput(EEFDigitalOutput.BCREXTMUXEN, 0)

    await bcr.SetParameter(SerialParameter.ReadTimeout, int(timeout * 1000))
    await bcr.ClearReadBuffer(timeout=1)
    if not bool(sw_trigger):
        await node.SetDigitalOutput(ntrig, 0)
    else:
        await bcr.Write(2, b'\x1B\x31')
        response = await bcr.Read(1, timeout=(timeout + 0.1))
        if response[0] < 1 or response[1] != 6:
            PyLogger.logger.error(f"Failed to receive sw_trigger success response {response}")

    try:
        response = await bcr.Read(length, timeout=(timeout + 0.1))
        length = response[0]
        barcode = response[1:(1 + length)]
    except BaseException as ex:
        PyLogger.logger.error(ex)

    if not bool(sw_trigger):
        await node.SetDigitalOutput(ntrig, 1)
        
    PyLogger.logger.info(f"Barcode: {barcode}")
    return barcode


async def bcr_read_until(bcr_name, termination='\r', timeout=5, sw_trigger=False) -> bytes:
    if isinstance(termination, str):
        termination = ord(termination)
    bcr = get_serial_endpoint(bcr_name)
    node = get_node_endpoint(serial_endpoints[bcr_name]['node'])
    ntrig = serial_endpoints[bcr_name]['ntrig']
    barcode = None
    if 'mux' in serial_endpoints[bcr_name]:
        mux = serial_endpoints[bcr_name]['mux']
        await node.SetDigitalOutput(EEFDigitalOutput.BCREXTMUX0, bool(mux & 1))
        await node.SetDigitalOutput(EEFDigitalOutput.BCREXTMUX1, bool(mux & 2))
        await node.SetDigitalOutput(EEFDigitalOutput.BCREXTMUXEN, 0)

    await bcr.SetParameter(SerialParameter.ReadTimeout, int(timeout * 1000))
    await bcr.ClearReadBuffer(timeout=1)
    if not bool(sw_trigger):
        await node.SetDigitalOutput(ntrig, 0)
    else:
        await bcr.Write(2, b'\x1B\x31')
        response = await bcr.Read(1, timeout=(timeout + 0.1))
        if response[0] < 1 or response[1] != 6:
            PyLogger.logger.error(f"Failed to receive sw_trigger success response {response}")

    try:
        response = await bcr.ReadUntil(termination, timeout=(timeout + 0.1))
        length = response[0]
        barcode = response[1:(1 + length)]
    except BaseException as ex:
        PyLogger.logger.error(ex)

    if not bool(sw_trigger):
        await node.SetDigitalOutput(ntrig, 1)
        
    PyLogger.logger.info(f"Barcode: {barcode}")
    return barcode


async def bcr_overflow(bcr_name):
    bcr = get_serial_endpoint(bcr_name)
    node = get_node_endpoint(serial_endpoints[bcr_name]['node'])
    ntrig = serial_endpoints[bcr_name]['ntrig']
    if 'mux' in serial_endpoints[bcr_name]:
        mux = serial_endpoints[bcr_name]['mux']
        await node.SetDigitalOutput(EEFDigitalOutput.BCREXTMUX0, bool(mux & 1))
        await node.SetDigitalOutput(EEFDigitalOutput.BCREXTMUX1, bool(mux & 2))
        await node.SetDigitalOutput(EEFDigitalOutput.BCREXTMUXEN, 0)

    await bcr.ClearReadBuffer(timeout=1)
    for i in range(100):
        await node.SetDigitalOutput(ntrig, 0)
        await asyncio.sleep(0.9)
        await node.SetDigitalOutput(ntrig, 1)
        await asyncio.sleep(0.1)
    
    PyLogger.logger.info(f"BCR Status: {await bcr.GetStatus(timeout=1)}")


# TRF Laser Tests ######################################################################################################

async def trf_send(command: str, timeout=1.0):
    serial = get_serial_endpoint('trf')
    command = command + '\r\n'
    await serial.SetParameter(SerialParameter.ReadTimeout, int(timeout * 1000))
    await serial.ClearReadBuffer(timeout=1)
    await serial.Write(len(command), command.encode('utf-8'), timeout=2)
    rxdata = ''
    try:
        response = await serial.ReadUntil(ord('\r'), timeout=(timeout + 0.1))
        length = response[0]
        rxdata = response[1:(1 + length)]
    except BaseException as ex:
        PyLogger.logger.error(ex)

    return bytes(rxdata).decode('utf-8')


# Temperature Control Tests ############################################################################################

async def tc_log(name, channel, interval, duration):
    tc = get_temperature_control_endpoint(name)
    with open('temperature_control_log.txt', 'a') as file:
        file.write('TIMESTAMP[s]  ; REFERENCE[FS] ; FEEDBACK[FS]  ; OUTPUT[FS]\n')
        start = time.perf_counter()
        timestamp = start
        while (timestamp - start) < duration:
            results = await asyncio.gather(
                tc.GetParameter(channel, TemperatureControlParameter.ReferenceValue),
                tc.GetFeedbackValue(channel),
                tc.GetOutputValue(channel),
                asyncio.sleep(interval)
            )
            reference = results[0][0]
            feedback  = results[1][0]
            output    = results[2][0]
            file.write(f"{timestamp:13.3f} ; {reference:+13.6f} ; {feedback:+13.6f} ; {output:+13.6f}\n")
            timestamp = time.perf_counter()
        
        file.write('\n')

    return f"tc_log() done"


async def tc_alpha_test():
    meas = get_measurement_endpoint()

    # Initialize Alpha TC
    await init('alpha_tc')
    # Set Laser Diode Current (≈2,5V)
    await meas.SetParameter(MeasurementParameter.AlphaLaserPower, 0.75)
    # Enable Laser Diode Current Supply
    await meas.SetParameter(MeasurementParameter.AlphaLaserEnable, 1)

    await seq_set_signal('alpha')       # Disable alpha laser
    await asyncio.sleep(10)             # Wait 10 s to stabalize the temperature
    await seq_clear_signal('alpha')     # Enable alpha laser
    await asyncio.sleep(20)             # Wait 10 s to stabalize the temperature
    await seq_set_signal('alpha')       # Disable alpha laser

    return f"tc_alpha_test() done"


async def tc_alpha_test_log():
    await init('eef')
    await asyncio.gather(
        tc_log('eef_tc', 0, 0, 50),
        tc_alpha_test()
    )

    return f"tc_alpha_test_log() done"


    
# Stack High Water Mark Tests ##########################################################################################

def get_endpoint_name(can_id):
    endpoint_dict = {
        **node_endpoints, **mover_endpoints, **corexy_mover_endpoints, **stacker_lift_mover_endpoints, 
        **measurement_endpoints, **serial_endpoints, **barcode_reader_endpoints, **system_control_endpoints,
        **fan_control_endpoints, **temperature_control_endpoints
    }
    for name in endpoint_dict:
        if endpoint_dict[name]['id'] == can_id:
            return name

    raise ValueError(f"can_id 0x{can_id:04X} is unknown")


async def get_stack_high_water_mark(node_name):
    node = get_node_endpoint(node_name)
    results = []
    for number in range(17):
        results.append(await node.GetStackHighWaterMark(number))
    await asyncio.sleep(0.5)
    for result in results:
        if (result[0] > 0):
            PyLogger.logger.info(f"High Water Mark {get_endpoint_name(result[0])}_{result[1]} : {result[2]}")
    await asyncio.sleep(0.5)
    return f"get_stack_high_water_mark() done"

async def trigger_n_times(frequency, time, flash_pwr=0.0, high_pwr=0):

    meas = get_measurement_endpoint()
    
    iterations = frequency*time
    duration_us = (((1/frequency)/2)*1000000)
    # Cancel if HighPower is set and Freqency above 500
    if frequency > 500 and high_pwr == 1:
        raise Exception(f"No HighPwr above 500Hz")
    
    node = get_node_endpoint('eef')
    
    # Set the Flash_Pwr if given
    if flash_pwr is not None:
        await node.SetAnalogOutput(12,flash_pwr)
        await asyncio.sleep(0.1)
    
    # Set HighPower 
    await node.SetDigitalOutput(9,high_pwr)
    await asyncio.sleep(0.1)


    ontime = int(100 * duration_us)
    offtime = int(1 / frequency * 1e8 - ontime)
    loop_a = int(iterations / 65536)
    loop_b = int(iterations % 65536)
    data = []
    if loop_a > 0:
        data.extend([
            0x07000000 | (loop_a - 1),      # Loop A Multiple of 2^16
            0x07000000 | (65536 - 1),       # Loop Inner
            0x7C000000 | (ontime - 1),      # Wait for offtime timer then start ontime timer
            0x02000000 | signals['flash'],  # Set Flash_Trg high
            0x7C000000 | (offtime - 1),     # Wait for ontime timer then start offtime timer
            0x03000000 | signals['flash'],  # Set Flash_Trg low
            0x05000000,                     # LoopEnd
            0x05000000,                     # LoopEnd
        ])
    if loop_b > 0:
        data.extend([
            0x07000000 | (loop_b - 1),      # Loop B Remainder
            0x7C000000 | (ontime - 1),      # Wait for offtime timer then start ontime timer
            0x02000000 | signals['flash'],  # Set Flash_Trg high
            0x7C000000 | (offtime - 1),     # Wait for ontime timer then start offtime timer
            0x03000000 | signals['flash'],  # Set Flash_Trg low
            0x05000000,                     # LoopEnd
        ])
    data.extend([
        0x00000000,                     # Stop
    ])
    await meas.WriteSequence(0, len(data), data, timeout=5)
    await meas.StartSequence(0)

